// Sistema de notícias com Google Sheets - Página Notícias
document.addEventListener('DOMContentLoaded', function() {
    // CORREÇÃO: Removida inicialização duplicada do menu mobile
    // O menu mobile já é inicializado pelo script.js principal
    // Duplicar a inicialização causa conflitos nos event listeners

    // Atualiza ano no rodapé
    updateCurrentYear();

    // Verifica se é a página de notícias e inicializa adequadamente
    if (window.location.pathname.includes('noticias.html')) {
        // Inicializa a paginação e carrega as notícias
        inicializarPaginacao();
        // Aguarda um pequeno delay para garantir que a paginação foi configurada
        setTimeout(function() {
            carregarNoticiasComPaginacao();
        }, 100);
    } else {
        // Se não é a página de notícias, carrega normalmente sem paginação
        carregarNoticiasGoogleSheets();
    }
});

// URL da planilha do Google Sheets
const GOOGLE_SHEETS_ID = '11GrxZcfsZzNYoLz0O65s_HSe-VTufYyo0CkElPW2zMQ';
const GOOGLE_SHEETS_URL = `https://docs.google.com/spreadsheets/d/${GOOGLE_SHEETS_ID}/export?format=csv`;

// Notícias de fallback para ambiente local
const noticiasBasicas = [
    {
        id: 'noticia-local-1',
        data: '27/03/2025',
        titulo: 'ATENÇÃO, ATENÇÃO - Assembleia de Prestação de Contas',
        categoria: 'CONVOCAÇÃO',
        descricao: 'Dia 31/03/2025 Segunda-feira<br>Primeira chamada às 18:00<br>Segunda chamada às 18:30<br><br>O Sindicato dos Securitários de MG convida todos os securitários para participarem da Assembleia de Prestação de Contas referente ao ano de 2024.',
        imagem: 'attached_assets/cropimage-20190117180000_1750362489712.jpeg'
    },
    {
        id: 'noticia-local-2',
        data: '26/03/2025',
        titulo: 'CAMPANHA SALARIAL 2025',
        categoria: 'CAMPANHA',
        descricao: 'O Sindicato dos Securitários conquistou uma importante vitória para a categoria na Campanha Salarial 2025. Acompanhe as negociações e conquistas obtidas para os trabalhadores do setor.',
        imagem: 'attached_assets/cropimage-20190117180339_1750362489712.jpeg'
    },
    {
        id: 'noticia-local-3',
        data: '20/02/2025',
        titulo: 'PLR DA BRADESCO SEGUROS FOI ASSINADA',
        categoria: 'ACORDO',
        descricao: 'AGORA, A LUTA É POR UMA CCT COM GANHOS REAIS! A PLR da Bradesco Seguros foi assinada com êxito. O sindicato continua trabalhando para garantir melhores condições na Convenção Coletiva de Trabalho.',
        imagem: 'attached_assets/cropimage-20190118112927_1750362489711.jpeg'
    },
    {
        id: 'noticia-local-4',
        data: '15/01/2025',
        titulo: 'Novos Convênios Disponíveis',
        categoria: 'BENEFÍCIOS',
        descricao: 'O sindicato firmou novos convênios para beneficiar os associados. Confira as novas parcerias em educação, saúde e lazer disponíveis para todos os securitários.',
        imagem: 'attached_assets/logo-20181210104822_1750362776018.png'
    },
    {
        id: 'noticia-local-5',
        data: '10/12/2024',
        titulo: 'Colônias de Férias em Funcionamento',
        categoria: 'LAZER',
        descricao: 'As colônias de férias do sindicato em Felixlândia, Nova Ponte e Jacaraípe estão em pleno funcionamento. Reserve já o seu período de descanso com a família.',
        imagem: 'attached_assets/cropimage-20200114133921_1750362489712.jpeg'
    }
];

// Variáveis de paginação
let noticiasPorPagina = 6; // Valor padrão
let paginaAtual = 1;
let paginacaoAtiva = true;

// Função para formatar data
function formatarData(dataString) {
    if (!dataString || !dataString.includes('/')) {
        return { completa: dataString, display: dataString };
    }

    const partes = dataString.split('/');
    if (partes.length !== 3) {
        return { completa: dataString, display: dataString };
    }

    const dia = partes[0].padStart(2, '0');
    const mes = partes[1].padStart(2, '0');
    const ano = partes[2];

    const meses = ['JAN', 'FEV', 'MAR', 'ABR', 'MAI', 'JUN',
                   'JUL', 'AGO', 'SET', 'OUT', 'NOV', 'DEZ'];
    const mesIndex = parseInt(partes[1]) - 1;
    const mesAbrev = meses[mesIndex] || 'JAN';

    return {
        completa: `${dia}/${mes}/${ano}`,
        display: `${parseInt(dia)} ${mesAbrev} ${ano}`
    };
}

// Função para criar HTML das notícias
function criarNoticiasHTML(noticias) {
    return noticias.map((noticia, index) => {
        const dataFormatada = formatarData(noticia.data);
        const imagemUrl = noticia.imagem || 'attached_assets/logo-20181210104822_1750362776018.png';
        const descricaoLimitada = noticia.descricao && noticia.descricao.length > 120 
            ? noticia.descricao.substring(0, 120) + '...' 
            : noticia.descricao || '';

        // Sanitiza os dados para JSON
        const noticiaSegura = {
            id: noticia.id,
            titulo: noticia.titulo.replace(/'/g, "\\'").replace(/"/g, '\\"'),
            data: dataFormatada.display,
            descricao: noticia.descricao.replace(/'/g, "\\'").replace(/"/g, '\\"'),
            imagem: imagemUrl,
            categoria: noticia.categoria || 'NOTÍCIA'
        };

        return `
            <article class="noticia-card" data-noticia-id="${noticia.id}" data-noticia-index="${index}">
                <div class="noticia-imagem">
                    <img src="${imagemUrl}" 
                         alt="${noticia.titulo}" 
                         loading="lazy"
                         data-original-url="${noticia.imagem || ''}"
                         onerror="handleImageError(this, '${noticia.titulo.replace(/'/g, "\\'")}');">
                </div>
                <div class="noticia-conteudo">
                    <h2 class="noticia-titulo">${noticia.titulo}</h2>
                    <div class="noticia-data">${dataFormatada.display}</div>
                    <div class="noticia-descricao">${descricaoLimitada}</div>
                </div>
            </article>
        `;
    }).join('');
}

// Função para detectar se está rodando localmente
function isLocalEnvironment() {
    return window.location.protocol === 'file:' || 
           window.location.hostname === 'localhost' || 
           window.location.hostname === '127.0.0.1' ||
           window.location.hostname === '';
}

// Função principal para carregar notícias do Google Sheets
async function carregarNoticiasGoogleSheets() {
    const loadingElement = document.getElementById('loading');
    const containerElement = document.getElementById('noticias-container');
    const errorElement = document.getElementById('error-message');

    try {
        console.log('📊 Carregando notícias...');

        // Mostra loading
        if (loadingElement) loadingElement.style.display = 'block';
        if (containerElement) containerElement.style.display = 'none';
        if (errorElement) errorElement.style.display = 'none';

        // Verifica se está rodando localmente
        if (isLocalEnvironment()) {
            console.log('🏠 Ambiente local detectado, usando notícias estáticas');
            // Esconde loading
            if (loadingElement) loadingElement.style.display = 'none';
            // Carrega notícias básicas diretamente
            carregarNoticiasBasicas();
            return;
        }

        console.log('🌐 Ambiente online detectado, tentando carregar do Google Sheets...');
        const response = await fetch(GOOGLE_SHEETS_URL);

        if (!response.ok) {
            throw new Error(`Erro HTTP: ${response.status}`);
        }

        const csvData = await response.text();
        console.log('📄 Tamanho dos dados CSV:', csvData.length, 'caracteres');
        const noticias = parseCSVToNoticias(csvData);

        console.log('🔍 Notícias extraídas:', noticias);

        // Esconde loading
        if (loadingElement) loadingElement.style.display = 'none';

        if (noticias.length > 0) {
            // Ordena por data (mais recentes primeiro)
            const noticiasOrdenadas = noticias.sort((a, b) => {
                if (!a.data || !b.data) return 0;
                const [diaA, mesA, anoA] = a.data.split('/');
                const [diaB, mesB, anoB] = b.data.split('/');
                const dataA = new Date(anoA, mesA - 1, diaA);
                const dataB = new Date(anoB, mesB - 1, diaB);
                return dataB - dataA;
            });

            // Renderiza na página (com ou sem paginação)
            if (containerElement) {
                if (paginacaoAtiva) {
                    // Se paginação está ativa, aplica paginação
                    const noticiasPagina = aplicarPaginacao(noticiasOrdenadas);
                    const noticiasHTML = criarNoticiasHTML(noticiasPagina);
                    containerElement.innerHTML = noticiasHTML;
                    containerElement.style.display = 'grid';

                    // Adiciona eventos de clique para notícias da página atual
                    adicionarEventosClique(noticiasPagina);
                } else {
                    // Se paginação não está ativa, mostra todas
                    const noticiasHTML = criarNoticiasHTML(noticiasOrdenadas);
                    containerElement.innerHTML = noticiasHTML;
                    containerElement.style.display = 'grid';

                    // Adiciona eventos de clique para todas as notícias
                    adicionarEventosClique(noticiasOrdenadas);
                }
            }

            console.log(`✅ ${noticias.length} notícias carregadas do Google Sheets!`);
        } else {
            throw new Error('Nenhuma notícia encontrada na planilha');
        }

    } catch (error) {
        console.error('❌ Erro ao carregar notícias:', error);

        // Esconde loading
        if (loadingElement) loadingElement.style.display = 'none';

        // Tenta carregar notícias básicas como fallback
        carregarNoticiasBasicas();
    }
}

// Função de fallback para carregar notícias básicas
function carregarNoticiasBasicas() {
    const containerElement = document.getElementById('noticias-container');
    const errorElement = document.getElementById('error-message');

    if (isLocalEnvironment()) {
        console.log('📋 Carregando notícias estáticas para ambiente local...');
    } else {
        console.log('📋 Carregando notícias básicas como fallback...');
    }

    // Ordena por data (mais recentes primeiro)
    const noticiasOrdenadas = [...noticiasBasicas].sort((a, b) => {
        if (!a.data || !b.data) return 0;
        const [diaA, mesA, anoA] = a.data.split('/');
        const [diaB, mesB, anoB] = b.data.split('/');
        const dataA = new Date(anoA, mesA - 1, diaA);
        const dataB = new Date(anoB, mesB - 1, diaB);
        return dataB - dataA;
    });

    // Renderiza na página (com ou sem paginação)
    if (containerElement) {
        if (paginacaoAtiva) {
            // Se paginação está ativa, aplica paginação
            const noticiasPagina = aplicarPaginacao(noticiasOrdenadas);
            const noticiasHTML = criarNoticiasHTML(noticiasPagina);
            containerElement.innerHTML = noticiasHTML;
            containerElement.style.display = 'grid';

            // Adiciona eventos de clique para notícias da página atual
            adicionarEventosClique(noticiasPagina);
        } else {
            // Se paginação não está ativa, mostra todas
            const noticiasHTML = criarNoticiasHTML(noticiasOrdenadas);
            containerElement.innerHTML = noticiasHTML;
            containerElement.style.display = 'grid';

            // Adiciona eventos de clique para todas as notícias
            adicionarEventosClique(noticiasOrdenadas);
        }
    }

    if (isLocalEnvironment()) {
        console.log('✅ Notícias estáticas carregadas para visualização local!');
    } else {
        console.log('✅ Notícias básicas carregadas como fallback!');
    }
}

// Regex para validar formato de data
const dateRegex = /^\d{1,2}\/\d{1,2}\/\d{4}$/;

// Função para converter CSV em array de notícias
function parseCSVToNoticias(csvData) {
    console.log('📄 Dados CSV recebidos (primeiras 500 chars):', csvData.substring(0, 500));

    const noticias = [];

    // Primeiro, processa o CSV considerando células com quebras de linha
    const processedLines = processCSVWithMultiline(csvData);
    const lines = processedLines.filter(line => line.trim() !== '');

    console.log('📊 Total de linhas encontradas:', lines.length);

    // Identifica se há cabeçalho
    let startIndex = 0;
    if (lines.length > 0) {
        const firstLine = lines[0].toLowerCase();
        if (firstLine.includes('data') || firstLine.includes('titulo') || firstLine.includes('título') || firstLine.includes('noticia')) {
            startIndex = 1;
            console.log('📋 Cabeçalho detectado, iniciando na linha 2');
        }
    }

    for (let i = startIndex; i < lines.length; i++) {
        const line = lines[i].trim();
        if (line && line.length > 10) {
            const columns = parseCSVLine(line);
            console.log(`📝 Linha ${i + 1}: ${columns.length} colunas encontradas`, columns);

            // Função para validar se uma string é uma data válida
            function isValidDate(dateString) {
                if (!dateString || typeof dateString !== 'string') return false;

                // Formato DD/MM/YYYY ou D/M/YYYY
                if (!dateRegex.test(dateString.trim())) return false;

                const [dia, mes, ano] = dateString.trim().split('/');
                const date = new Date(ano, mes - 1, dia);

                return date.getFullYear() == ano && 
                       date.getMonth() == (mes - 1) && 
                       date.getDate() == dia;
            }

            // Função para validar se uma string parece um título válido
            function isValidTitle(titleString) {
                if (!titleString || typeof titleString !== 'string') return false;
                const cleaned = titleString.trim();
                // Título deve ter pelo menos 3 caracteres e não ser só números ou data
                return cleaned.length >= 3 && 
                       !dateRegex.test(cleaned) && 
                       !/^\d+$/.test(cleaned);
            }

            let data, titulo, descricao, foto;
            let found = false;

            // Procura pela data e título válidos
            for (let j = 0; j < Math.min(columns.length, 5); j++) {
                if (found) break;

                if (isValidDate(columns[j])) {
                    for (let k = j + 1; k < Math.min(columns.length, j + 4); k++) {
                        if (isValidTitle(columns[k])) {
                            data = columns[j].trim();
                            titulo = columns[k].trim();

                            if (k + 1 < columns.length && columns[k + 1]) {
                                descricao = columns[k + 1].trim();
                            }
                            if (k + 2 < columns.length && columns[k + 2]) {
                                foto = columns[k + 2].trim();
                            }

                            found = true;
                            break;
                        }
                    }
                }
            }

            if (found && data && titulo) {
                const noticia = {
                    id: 'noticia-sheets-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9),
                    data: data,
                    titulo: titulo,
                    descricao: descricao || 'Descrição não disponível',
                    imagem: processarURLImagem(foto),
                    categoria: 'NOTÍCIA'
                };

                noticias.push(noticia);
            }
        }
    }

    return noticias;
}

// Função para processar linha CSV (lida com vírgulas dentro de aspas e quebras de linha)
function parseCSVLine(line) {
    const result = [];
    let current = '';
    let inQuotes = false;
    let quoteChar = null;

    for (let i = 0; i < line.length; i++) {
        const char = line[i];

        if ((char === '"' || char === "'") && !inQuotes) {
            inQuotes = true;
            quoteChar = char;
        } else if (char === quoteChar && inQuotes) {
            // Verifica se é uma aspa de escape (duas aspas seguidas)
            if (i + 1 < line.length && line[i + 1] === quoteChar) {
                current += char;
                i++; // Pula a próxima aspa
            } else {
                inQuotes = false;
                quoteChar = null;
            }
        } else if (char === ',' && !inQuotes) {
            result.push(current.trim());
            current = '';
        } else {
            current += char;
        }
    }

    result.push(current.trim());

    // Remove aspas externas e limpa espaços
    return result.map(col => {
        let cleaned = col.trim();
        // Remove aspas externas apenas se a string começar e terminar com a mesma aspa
        if ((cleaned.startsWith('"') && cleaned.endsWith('"')) || 
            (cleaned.startsWith("'") && cleaned.endsWith("'"))) {
            cleaned = cleaned.slice(1, -1);
        }
        // Converte quebras de linha em HTML
        cleaned = cleaned.replace(/\\n/g, '\n').replace(/\n/g, '<br>');
        return cleaned.trim();
    });
}

// Função para processar URL da imagem do Google Drive
function processarURLImagem(urlString) {
    console.log('🖼️ Processando URL da imagem:', urlString);

    if (!urlString || urlString.trim() === '') {
        console.log('⚠️ URL vazia, usando imagem padrão');
        return 'attached_assets/logo-20181210104822_1750362776018.png';
    }

    let url = urlString.trim();

    // Remove aspas se existirem
    url = url.replace(/^["']|["']$/g, '');

    // Remove função HIPERLINK se existir
    if (url.includes('=HIPERLINK(')) {
        const match = url.match(/=HIPERLINK\("([^"]+)"/);
        if (match && match[1]) {
            url = match[1];
            console.log('🔗 URL extraída de HIPERLINK:', url);
        }
    }

    let fileId = null;

    // Extrai ID do Google Drive
    if (url.includes('drive.google.com/open?id=')) {
        const match = url.match(/[?&]id=([a-zA-Z0-9-_]+)/);
        if (match && match[1]) {
            fileId = match[1];
            console.log('📎 ID extraído de open?id:', fileId);
        }
    } else if (url.includes('drive.google.com/file/d/')) {
        const match = url.match(/\/file\/d\/([a-zA-Z0-9-_]+)/);
        if (match && match[1]) {
            fileId = match[1];
            console.log('📎 ID extraído de file/d:', fileId);
        }
    } else if (url.includes('drive.google.com/uc?')) {
        const match = url.match(/[?&]id=([a-zA-Z0-9-_]+)/);
        if (match && match[1]) {
            fileId = match[1];
            console.log('📎 ID extraído de uc?id:', fileId);
        }
    }

    // Se encontrou ID do Google Drive, gera URL otimizada
    if (fileId) {
        const selectedUrl = `https://drive.google.com/uc?export=view&id=${fileId}`;
        console.log('✅ URL convertida para:', selectedUrl);
        console.log('🆔 ID do arquivo:', fileId);
        return selectedUrl;
    }

    console.log('ℹ️ URL não é do Google Drive, usando original:', url);
    // Se não for do Google Drive, retorna a URL original
    return url || 'attached_assets/logo-20181210104822_1750362776018.png';
}

// Função para processar CSV com células que contêm quebras de linha
function processCSVWithMultiline(csvData) {
    const lines = [];
    const rawLines = csvData.split('\n');
    let currentLine = '';
    let inQuotes = false;
    let quoteChar = null;

    for (let i = 0; i < rawLines.length; i++) {
        const line = rawLines[i];

        for (let j = 0; j < line.length; j++) {
            const char = line[j];

            if ((char === '"' || char === "'") && !inQuotes) {
                inQuotes = true;
                quoteChar = char;
                currentLine += char;
            } else if (char === quoteChar && inQuotes) {
                if (j + 1 < line.length && line[j + 1] === quoteChar) {
                    currentLine += char + char;
                    j++; // Pula a próxima aspa
                } else {
                    inQuotes = false;
                    quoteChar = null;
                    currentLine += char;
                }
            } else {
                currentLine += char;
            }
        }

        // Se não estamos dentro de aspas, a linha está completa
        if (!inQuotes) {
            if (currentLine.trim() !== '') {
                lines.push(currentLine);
            }
            currentLine = '';
        } else {
            // Se estamos dentro de aspas, adiciona quebra de linha e continua
            currentLine += '\\n';
        }
    }

    // Adiciona a última linha se houver
    if (currentLine.trim() !== '') {
        lines.push(currentLine);
    }

    return lines;
}

// Função para tratar erro de carregamento de imagens
function handleImageError(img, titulo) {
    console.error('❌ Erro ao carregar imagem para:', titulo);
    console.error('🔗 URL que falhou:', img.src);

    // Se já tentou a imagem padrão, não tenta novamente
    if (img.src.includes('logo-20181210104822_1750362776018.png')) {
        console.error('⚠️ Até a imagem padrão falhou para:', titulo);
        return;
    }

    // Obtém o ID original se for do Google Drive
    const originalUrl = img.getAttribute('data-original-url') || img.src;
    let fileId = null;

    // Extrai ID do Google Drive
    if (originalUrl.includes('drive.google.com')) {
        const idMatch = originalUrl.match(/[?&]id=([a-zA-Z0-9-_]+)/);
        if (idMatch && idMatch[1]) {
            fileId = idMatch[1];
        }
    }

    // Se tem ID, tenta formatos alternativos
    if (fileId && !img.hasAttribute('data-tried-alternatives')) {
        img.setAttribute('data-tried-alternatives', 'true');

        const alternativeFormats = [
            `https://drive.usercontent.google.com/download?id=${fileId}&export=view`,
            `https://lh3.googleusercontent.com/d/${fileId}`,
            `https://drive.google.com/uc?id=${fileId}`,
            `https://drive.google.com/thumbnail?id=${fileId}&sz=w400`
        ];

        console.log('🔄 Tentando formatos alternativos para:', titulo);
        tryAlternativeFormats(img, alternativeFormats, 0, titulo);
    } else {
        // Tenta a imagem padrão
        img.onerror = null; // Remove o handler para evitar loop
        img.src = 'attached_assets/logo-20181210104822_1750362776018.png';
        console.log('🔄 Usando imagem padrão para:', titulo);
    }
}

// Função para tentar formatos alternativos
function tryAlternativeFormats(img, formats, index, titulo) {
    if (index >= formats.length) {
        // Se todos os formatos falharam, usa imagem padrão
        img.onerror = null;
        img.src = 'attached_assets/logo-20181210104822_1750362776018.png';
        console.log('🔄 Todos os formatos falharam, usando imagem padrão para:', titulo);
        return;
    }

    const currentFormat = formats[index];
    console.log(`🔄 Tentando formato ${index + 1}/${formats.length}:`, currentFormat);

    img.onerror = function() {
        console.error(`❌ Formato ${index + 1} falhou:`, currentFormat);
        tryAlternativeFormats(img, formats, index + 1, titulo);
    };

    img.onload = function() {
        console.log('✅ Sucesso com formato:', currentFormat);
        img.onerror = null;
        img.onload = null;
    };

    img.src = currentFormat;
}

// Função para adicionar eventos de clique nas notícias
function adicionarEventosClique(noticias) {
    console.log('🎯 Adicionando eventos de clique para', noticias.length, 'notícias');

    // Remove listeners antigos se existirem
    document.querySelectorAll('.noticia-card').forEach(card => {
        const novoCard = card.cloneNode(true);
        card.parentNode.replaceChild(novoCard, card);
    });

    // Adiciona novos listeners
    document.querySelectorAll('.noticia-card').forEach((card, index) => {
        card.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();

            const noticiaIndex = parseInt(card.getAttribute('data-noticia-index'));
            const noticia = noticias[noticiaIndex];

            if (noticia) {
                console.log('🎯 Clique na notícia:', noticia.titulo);
                expandirCardNoticia(noticia);
            } else {
                console.error('❌ Notícia não encontrada no índice:', noticiaIndex);
            }
        });

        // Adiciona cursor pointer
        card.style.cursor = 'pointer';
    });

    console.log('✅ Eventos de clique adicionados para todas as notícias');
}

// Função para atualizar ano
function updateCurrentYear() {
    const yearElement = document.getElementById('current-year');
    if (yearElement) {
        yearElement.textContent = new Date().getFullYear();
    }
}

// Função para expandir card como modal - copiada da home
function expandirCardNoticia(noticia) {
    console.log('🎯 Criando modal expandido para:', noticia.titulo);

    // Remove overlay existente se houver
    const overlayExistente = document.querySelector('.noticia-overlay');
    if (overlayExistente) {
        overlayExistente.remove();
    }

    // Cria o overlay
    const overlay = document.createElement('div');
    overlay.className = 'noticia-overlay';

    // Cria o card expandido
    const cardExpandido = document.createElement('div');
    cardExpandido.className = 'noticia-card-expandido';

    // Conteúdo do card expandido
    cardExpandido.innerHTML = `
        <div class="noticia-header-expandido">
            <button class="btn-fechar-noticia" onclick="fecharCardNoticia()">×</button>
            <div class="noticia-data-expandido">${noticia.data}</div>
            <h2 class="noticia-titulo-expandido">${noticia.titulo}</h2>
        </div>

        ${noticia.imagem && !noticia.imagem.includes('logo-20181210104822') ? `
            <div class="noticia-imagem-expandido">
                <img src="${noticia.imagem}" 
                     alt="${noticia.titulo}" 
                     data-original-url="${noticia.imagem}"
                     onerror="handleImageErrorExpandido(this, '${noticia.titulo}')">
            </div>
        ` : ''}

        <div class="noticia-conteudo-expandido">
            <div class="noticia-categoria-expandido">${noticia.categoria}</div>
            <div class="noticia-descricao-expandido">
                ${noticia.descricao}
            </div>
        </div>
    `;

    overlay.appendChild(cardExpandido);
    document.body.appendChild(overlay);

    // Adiciona evento para fechar ao clicar no overlay
    overlay.addEventListener('click', function(e) {
        if (e.target === overlay) {
            fecharCardNoticia();
        }
    });

    // Adiciona evento para fechar com ESC
    const escapeHandler = function(e) {
        if (e.key === 'Escape') {
            fecharCardNoticia();
            document.removeEventListener('keydown', escapeHandler);
        }
    };
    document.addEventListener('keydown', escapeHandler);

    // Anima a entrada
    setTimeout(() => {
        overlay.classList.add('ativo');
        cardExpandido.classList.add('ativo');
    }, 10);
}

// Função para fechar o card expandido
function fecharCardNoticia() {
    const overlay = document.querySelector('.noticia-overlay');
    if (overlay) {
        overlay.classList.remove('ativo');
        setTimeout(() => {
            overlay.remove();
        }, 300);
    }
}

// Função para tratar erro de carregamento de imagens no modal
function handleImageErrorExpandido(img, titulo) {
    console.error('❌ Erro ao carregar imagem expandida para:', titulo);
    console.error('🔗 URL que falhou:', img.src);

    // Obtém o URL original se for do Google Drive
    const originalUrl = img.getAttribute('data-original-url') || img.src;
    let fileId = null;

    // Extrai ID do Google Drive
    if (originalUrl.includes('drive.google.com')) {
        const patterns = [
            /[?&]id=([a-zA-Z0-9-_]+)/,
            /\/file\/d\/([a-zA-Z0-9-_]+)/,
            /\/d\/([a-zA-Z0-9-_]+)/
        ];

        for (const pattern of patterns) {
            const match = originalUrl.match(pattern);
            if (match && match[1]) {
                fileId = match[1];
                break;
            }
        }
    }

    // Se tem ID, tenta formatos alternativos
    if (fileId && !img.hasAttribute('data-tried-alternatives-expandido')) {
        img.setAttribute('data-tried-alternatives-expandido', 'true');

        const alternativeFormats = [
            `https://drive.usercontent.google.com/download?id=${fileId}&export=view`,
            `https://lh3.googleusercontent.com/d/${fileId}`,
            `https://drive.google.com/uc?id=${fileId}`,
            `https://drive.google.com/thumbnail?id=${fileId}&sz=w800`
        ];

        console.log('🔄 Tentando formatos alternativos no modal das notícias para:', titulo);
        tryAlternativeFormatsExpandido(img, alternativeFormats, 0, titulo);
    } else {
        // Remove a imagem se não conseguir carregar
        const imagemContainer = img.parentElement;
        if (imagemContainer && imagemContainer.classList.contains('noticia-imagem-expandido')) {
            imagemContainer.style.display = 'none';
        }
        console.log('🔄 Ocultando imagem no modal das notícias para:', titulo);
    }
}

// Função para tentar formatos alternativos no modal
function tryAlternativeFormatsExpandido(img, formats, index, titulo) {
    if (index >= formats.length) {
        const imagemContainer = img.parentElement;
        if (imagemContainer && imagemContainer.classList.contains('noticia-imagem-expandido')) {
            imagemContainer.style.display = 'none';
        }
        console.log('🔄 Todos os formatos falharam no modal das notícias, ocultando imagem para:', titulo);
        return;
    }

    const currentFormat = formats[index];
    console.log(`🔄 Tentando formato ${index + 1}/${formats.length} no modal das notícias:`, currentFormat);

    img.onerror = function() {
        console.error(`❌ Formato ${index + 1} falhou no modal das notícias:`, currentFormat);
        tryAlternativeFormatsExpandido(img, formats, index + 1, titulo);
    };

    img.onload = function() {
        console.log('✅ Sucesso com formato no modal das notícias:', currentFormat);
        img.onerror = null;
        img.onload = null;
    };

    img.src = currentFormat;
}

// Função para aplicar a paginação
function aplicarPaginacao(noticias) {
    paginaAtual = Number(paginaAtual);
    noticiasPorPagina = Number(noticiasPorPagina);

    const totalNoticias = noticias.length;
    const totalPaginas = Math.ceil(totalNoticias / noticiasPorPagina);

    // Ajusta a página atual se estiver fora dos limites
    if (paginaAtual < 1) {
        paginaAtual = 1;
    } else if (paginaAtual > totalPaginas) {
        paginaAtual = totalPaginas;
    }

    const startIndex = (paginaAtual - 1) * noticiasPorPagina;
    const endIndex = startIndex + noticiasPorPagina;

    const noticiasPagina = noticias.slice(startIndex, endIndex);

    // Atualiza os botões de paginação
    atualizarBotoesPaginacao(totalPaginas);

    return noticiasPagina;
}

// Função para atualizar os botões de paginação
function atualizarBotoesPaginacao(totalPaginas) {
    const paginacaoContainer = document.getElementById('paginacao-container');

    // Verifica se o container de paginação existe
    if (!paginacaoContainer) {
        console.warn('Container de paginação não encontrado.');
        return;
    }

    // Se só há uma página, esconde a paginação
    if (totalPaginas <= 1) {
        paginacaoContainer.style.display = 'none';
        return;
    }

    paginacaoContainer.style.display = 'flex';

    // Limpa o container de paginação
    paginacaoContainer.innerHTML = '';

    // Cria o botão "Anterior"
    const btnAnterior = document.createElement('button');
    btnAnterior.textContent = '← Anterior';
    btnAnterior.classList.add('btn-paginacao');
    btnAnterior.disabled = paginaAtual === 1;
    btnAnterior.addEventListener('click', function() {
        paginaAtual--;
        carregarNoticiasComPaginacao();
    });
    paginacaoContainer.appendChild(btnAnterior);

    // Adiciona a numeração das páginas
    const mostrarNumeros = totalPaginas <= 7; // Exibe números se não houver muitas páginas

    if (mostrarNumeros) {
        for (let i = 1; i <= totalPaginas; i++) {
            const btnNumero = document.createElement('button');
            btnNumero.textContent = i;
            btnNumero.classList.add('btn-paginacao');
            if (i === paginaAtual) {
                btnNumero.classList.add('ativo');
            } else {
                btnNumero.addEventListener('click', function() {
                    paginaAtual = i;
                    carregarNoticiasComPaginacao();
                });
            }
            paginacaoContainer.appendChild(btnNumero);
        }
    } else {
        // Se houver muitas páginas, adiciona apenas alguns números e "..."
        if (paginaAtual > 3) {
            // Adiciona a primeira página
            const btnNumero1 = document.createElement('button');
            btnNumero1.textContent = 1;
            btnNumero1.classList.add('btn-paginacao');
            btnNumero1.addEventListener('click', function() {
                paginaAtual = 1;
                carregarNoticiasComPaginacao();
            });
            paginacaoContainer.appendChild(btnNumero1);

            // Adiciona "..."
            const spanDots1 = document.createElement('span');
            spanDots1.textContent = '...';
            spanDots1.classList.add('dots-paginacao');
            paginacaoContainer.appendChild(spanDots1);
        }

        // Adiciona as páginas próximas à atual
        let startPage = Math.max(1, paginaAtual - 1);
        let endPage = Math.min(totalPaginas, paginaAtual + 1);

        for (let i = startPage; i <= endPage; i++) {
            const btnNumero = document.createElement('button');
            btnNumero.textContent = i;
            btnNumero.classList.add('btn-paginacao');
            if (i === paginaAtual) {
                btnNumero.classList.add('ativo');
            } else {
                btnNumero.addEventListener('click', function() {
                    paginaAtual = i;
                    carregarNoticiasComPaginacao();
                });
            }
            paginacaoContainer.appendChild(btnNumero);
        }

        if (paginaAtual < totalPaginas - 2) {
            // Adiciona "..."
            const spanDots2 = document.createElement('span');
            spanDots2.textContent = '...';
            spanDots2.classList.add('dots-paginacao');
            paginacaoContainer.appendChild(spanDots2);

            // Adiciona a última página
            const btnNumeroLast = document.createElement('button');
            btnNumeroLast.textContent = totalPaginas;
            btnNumeroLast.classList.add('btn-paginacao');
            btnNumeroLast.addEventListener('click', function() {
                paginaAtual = totalPaginas;
                carregarNoticiasComPaginacao();
            });
            paginacaoContainer.appendChild(btnNumeroLast);
        }
    }

    // Cria o botão "Próxima"
    const btnProxima = document.createElement('button');
    btnProxima.textContent = 'Próxima →';
    btnProxima.classList.add('btn-paginacao');
    btnProxima.disabled = paginaAtual === totalPaginas;
    btnProxima.addEventListener('click', function() {
        paginaAtual++;
        carregarNoticiasComPaginacao();
    });
    paginacaoContainer.appendChild(btnProxima);
}

// Função para carregar notícias com paginação
function carregarNoticiasComPaginacao() {
    console.log('Carregando notícias com paginação. Página atual:', paginaAtual, 'Itens por página:', noticiasPorPagina);
    
    const containerElement = document.getElementById('noticias-container');
    const loadingElement = document.getElementById('loading');

    if (containerElement && loadingElement) {
        // Só mostra loading se não há conteúdo
        if (!containerElement.hasChildNodes() || containerElement.children.length === 0) {
            loadingElement.style.display = 'block';
            containerElement.style.display = 'none';
        }

        // Carrega as notícias diretamente
        if (isLocalEnvironment()) {
            carregarNoticiasBasicas();
        } else {
            carregarNoticiasGoogleSheets();
        }
    } else {
        console.warn('Elementos necessários não encontrados para carregar notícias com paginação.');
    }
}

// Função para inicializar a paginação (executada apenas em noticias.html)
function inicializarPaginacao() {
    if (window.location.pathname.includes('noticias.html')) {
        console.log('Página de notícias detectada. Inicializando paginação.');
        paginacaoAtiva = true; // Ativa a paginação
        
        // Verifica se já foi inicializada para evitar duplicação
        if (document.getElementById('paginacao-container')) {
            console.log('Paginação já inicializada.');
            return;
        }
        
        // Adiciona o container da paginação
        const containerNoticias = document.getElementById('container-noticias');
        if (containerNoticias) {
            // Cria o controle de seleção de itens por página
            const controlePaginacao = document.createElement('div');
            controlePaginacao.className = 'controle-paginacao';
            controlePaginacao.innerHTML = `
                <div class="items-per-page">
                    <label for="noticias-por-pagina">Mostrar:</label>
                    <select id="noticias-por-pagina" class="select-noticias-por-pagina">
                        <option value="6">6 por página</option>
                        <option value="12">12 por página</option>
                        <option value="18">18 por página</option>
                        <option value="24">24 por página</option>
                    </select>
                </div>
            `;
            
            // Cria o container da paginação
            const paginacaoContainer = document.createElement('div');
            paginacaoContainer.id = 'paginacao-container';
            paginacaoContainer.className = 'paginacao-container';
            
            // Adiciona os elementos ao container principal
            containerNoticias.appendChild(controlePaginacao);
            containerNoticias.appendChild(paginacaoContainer);

            // Configura o evento do select
            const selectNoticiasPorPagina = document.getElementById('noticias-por-pagina');
            selectNoticiasPorPagina.value = noticiasPorPagina; // Define o valor padrão
            selectNoticiasPorPagina.addEventListener('change', function() {
                noticiasPorPagina = parseInt(this.value);
                paginaAtual = 1; // Reseta para a primeira página
                carregarNoticiasComPaginacao();
            });
            
            console.log('Paginação inicializada com sucesso. Itens por página:', noticiasPorPagina);
        } else {
            console.warn('Container de notícias não encontrado.');
        }
    } else {
        console.log('Não é a página de notícias. Paginação não inicializada.');
        paginacaoAtiva = false; // Desativa a paginação
    }
}