// Aguarda o carregamento completo da página
document.addEventListener('DOMContentLoaded', function() {

    // Configuração do menu mobile
    initMobileMenu();

    // Configuração do carrossel
    initCarousel();

    // Configuração do carrossel de benefícios
    initBeneficiosCarousel();

    // Atualiza o ano no rodapé
    updateCurrentYear();

    // Configuração da navegação suave
    initSmoothScrolling();

    // Carrega notícias na página inicial
    setTimeout(carregarNoticiasHome, 500);
});

/**
 * ===== NOVO SISTEMA DE MENU MOBILE - MODULAR E LIMPO =====
 * Sistema completo de navegação mobile com overlay lateral
 */
function initMobileMenu() {
    // Verifica se já existe o menu mobile, se não, cria
    createMobileMenuStructure();

    // Elementos do menu mobile
    const mobileToggle = document.querySelector('.mobile-menu-toggle');
    const mobileOverlay = document.querySelector('.mobile-nav-overlay');
    const mobileContainer = document.querySelector('.mobile-nav-container');
    const mobileClose = document.querySelector('.mobile-nav-close');
    const mobileDropdowns = document.querySelectorAll('.mobile-nav-dropdown-toggle');
    const mobileLinks = document.querySelectorAll('.mobile-nav-link');

    if (!mobileToggle || !mobileOverlay || !mobileContainer) {
        console.warn('Elementos do menu mobile não encontrados');
        return;
    }

    // Estado do menu
    let isMenuOpen = false;

    /**
     * Abre o menu mobile
     */
    function openMobileMenu() {
        isMenuOpen = true;
        mobileToggle.classList.add('mobile-active');
        mobileOverlay.classList.add('mobile-active');
        mobileContainer.classList.add('mobile-active');
        document.body.style.overflow = 'hidden';

        // Foco no botão de fechar para acessibilidade
        setTimeout(() => {
            if (mobileClose) mobileClose.focus();
        }, 300);
    }

    /**
     * Fecha o menu mobile
     */
    function closeMobileMenu() {
        isMenuOpen = false;
        mobileToggle.classList.remove('mobile-active');
        mobileOverlay.classList.remove('mobile-active');
        mobileContainer.classList.remove('mobile-active');
        document.body.style.overflow = '';

        // Fecha todos os dropdowns abertos
        document.querySelectorAll('.mobile-nav-dropdown').forEach(dropdown => {
            dropdown.classList.remove('mobile-dropdown-open');
        });
    }

    /**
     * Toggle do menu mobile
     */
    function toggleMobileMenu(e) {
        e.preventDefault();
        e.stopPropagation();

        if (isMenuOpen) {
            closeMobileMenu();
        } else {
            openMobileMenu();
        }
    }

    // Event listeners para abrir/fechar menu
    if (mobileToggle) {
        mobileToggle.addEventListener('click', toggleMobileMenu);
        mobileToggle.addEventListener('touchstart', function(e) {
            e.preventDefault();
            toggleMobileMenu(e);
        });
    }

    if (mobileClose) {
        mobileClose.addEventListener('click', closeMobileMenu);
    }

    if (mobileOverlay) {
        mobileOverlay.addEventListener('click', closeMobileMenu);
    }

    // Event listeners para dropdowns
    mobileDropdowns.forEach(dropdownToggle => {
        dropdownToggle.addEventListener('click', function(e) {
            e.preventDefault();
            const dropdown = this.closest('.mobile-nav-dropdown');

            // Fecha outros dropdowns
            document.querySelectorAll('.mobile-nav-dropdown').forEach(otherDropdown => {
                if (otherDropdown !== dropdown) {
                    otherDropdown.classList.remove('mobile-dropdown-open');
                }
            });

            // Toggle do dropdown atual
            dropdown.classList.toggle('mobile-dropdown-open');
        });
    });

    // Event listeners para links do menu
    mobileLinks.forEach(link => {
        link.addEventListener('click', function() {
            // Fecha o menu com um pequeno delay para permitir navegação
            setTimeout(closeMobileMenu, 150);
        });
    });

    // Fecha menu com ESC
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && isMenuOpen) {
            closeMobileMenu();
        }
    });

    // Fecha menu ao redimensionar para desktop
    window.addEventListener('resize', function() {
        if (window.innerWidth > 768 && isMenuOpen) {
            closeMobileMenu();
        }
    });

    console.log('✅ Novo sistema de menu mobile inicializado com sucesso');
}

/**
 * Cria a estrutura HTML do menu mobile se não existir
 */
function createMobileMenuStructure() {
    console.log('🔧 Criando estrutura do menu mobile...');

    // FORÇA a remoção completa de qualquer menu mobile existente
    const existingOverlay = document.querySelector('.mobile-nav-overlay');
    const existingContainer = document.querySelector('.mobile-nav-container');
    if (existingOverlay) {
        existingOverlay.remove();
        console.log('🗑️ Menu mobile existente removido (overlay)');
    }
    if (existingContainer) {
        existingContainer.remove();
        console.log('🗑️ Menu mobile existente removido (container)');
    }

    // Remove também qualquer elemento órfão que possa estar interferindo
    document.querySelectorAll('[class*="mobile-nav"]').forEach(el => {
        if (!el.classList.contains('mobile-menu-toggle')) {
            el.remove();
        }
    });

    // CORREÇÃO: Substitui o botão hambúrguer existente (sistema antigo) pelo novo
    // Isso resolve o conflito na página de notícias onde ambos os sistemas coexistiam
    const oldToggle = document.getElementById('navbar-toggle');
    if (oldToggle) {
        const newToggle = document.createElement('button');
        newToggle.className = 'mobile-menu-toggle';
        newToggle.setAttribute('aria-label', 'Abrir menu mobile');
        newToggle.innerHTML = `
            <span class="hamburger-line"></span>
            <span class="hamburger-line"></span>
            <span class="hamburger-line"></span>
        `;
        oldToggle.parentNode.replaceChild(newToggle, oldToggle);
        console.log('🔧 Botão de menu antigo substituído pelo novo sistema modular');
    }


    // Lista completa de itens do submenu "Acordos e Convenções"
    const acordosSubmenuItems = [
        // ANOS HISTÓRICOS (2009-2019)
        { link: `${basePath}acordos/acordos-2009.html`, label: '2009' },
        { link: `${basePath}acordos/acordos-2010.html`, label: '2010' },
        { link: `${basePath}acordos/acordos-2011.html`, label: '2011' },
        { link: `${basePath}acordos/acordos-2012.html`, label: '2012' },
        { link: `${basePath}acordos/acordos-2013.html`, label: '2013' },
        { link: `${basePath}acordos/acordos-2014.html`, label: '2014' },
        { link: `${basePath}acordos/acordos-2015.html`, label: '2015' },
        { link: `${basePath}acordos/acordos-2016.html`, label: '2016' },
        { link: `${basePath}acordos/acordos-2017.html`, label: '2017' },
        { link: `${basePath}acordos/acordos-2018.html`, label: '2018' },
        { link: `${basePath}acordos/acordos-2019.html`, label: '2019' },
        // ANOS RECENTES (2020-2024) - OBRIGATÓRIOS
        { link: `${basePath}acordos/acordos-2020.html`, label: '2020' },
        { link: `${basePath}acordos/acordos-2021.html`, label: '2021' },
        { link: `${basePath}acordos/acordos-2022.html`, label: '2022' },
        { link: `${basePath}acordos/acordos-2023.html`, label: '2023' },
        { link: `${basePath}acordos/acordos-2024.html`, label: '2024' },
        // PÁGINAS ESPECIAIS - OBRIGATÓRIAS
        { link: `${basePath}acordos/cct-acordos-convencoes.html`, label: 'CCT - Acordos e Convenções' },
        { link: `${basePath}acordos/cct-plr-seguradoras-2024.html`, label: 'PLR 2024' },
        { link: `${basePath}acordos/convencao-coletiva.html`, label: 'Convenção Coletiva' }
    ];

    console.log('📋 Total de itens no submenu Acordos:', acordosSubmenuItems.length);
    console.log('📝 Lista completa dos itens:', acordosSubmenuItems.map(item => item.label));

    // Gera HTML dos itens do submenu
    const acordosSubmenuHTML = acordosSubmenuItems.map((item, index) => {
        console.log(`📄 Item ${index + 1}: ${item.label} -> ${item.link}`);
        return `<li><a href="${item.link}" class="mobile-nav-link">${item.label}</a></li>`;
    }).join('\n                        ');

    // Cria a estrutura do menu mobile
    const mobileMenuHTML = `
        <!-- Overlay do menu mobile -->
        <div class="mobile-nav-overlay"></div>

        <!-- Container do menu mobile -->
        <div class="mobile-nav-container">
            <!-- Header do menu -->
            <div class="mobile-nav-header">
                <img src="attached_assets/logo-20181210104822_1750362776018.png" 
                     alt="Logo Sindicato" class="mobile-nav-logo">
                <button class="mobile-nav-close" aria-label="Fechar menu">&times;</button>
            </div>

            <!-- Lista de navegação -->
            <ul class="mobile-nav-list">
                <li class="mobile-nav-item">
                    <a href="${basePath}index.html" class="mobile-nav-link">Home</a>
                </li>
                <li class="mobile-nav-item">
                    <a href="${basePath}sindicato.html" class="mobile-nav-link">O Sindicato</a>
                </li>
                <li class="mobile-nav-item">
                    <a href="${basePath}associe-se.html" class="mobile-nav-link">Associe-se</a>
                </li>
                <li class="mobile-nav-item mobile-nav-dropdown">
                    <button class="mobile-nav-dropdown-toggle">
                        Acordos e Convenções
                        <span class="mobile-nav-dropdown-icon">▼</span>
                    </button>
                    <ul class="mobile-nav-submenu">
                        ${acordosSubmenuHTML}
                    </ul>
                </li>
                <li class="mobile-nav-item">
                    <a href="${basePath}noticias.html" class="mobile-nav-link">Notícias</a>
                </li>
                <li class="mobile-nav-item">
                    <a href="${basePath}beneficios.html" class="mobile-nav-link">Benefícios - Convênios</a>
                </li>
                <li class="mobile-nav-item">
                    <a href="${basePath}lazer.html" class="mobile-nav-link">Lazer</a>
                </li>
                <li class="mobile-nav-item mobile-nav-dropdown">
                    <button class="mobile-nav-dropdown-toggle">
                        Fotos
                        <span class="mobile-nav-dropdown-icon">▼</span>
                    </button>
                    <ul class="mobile-nav-submenu">
                        <li><a href="${basePath}fotos/fotos-clube-asf.html" class="mobile-nav-link">Clube em parceria com ASF</a></li>
                        <li><a href="${basePath}fotos/fotos-clube-celp.html" class="mobile-nav-link">Clube CELP</a></li>
                        <li><a href="${basePath}fotos/fotos-felixlandia.html" class="mobile-nav-link">Colônia de Férias Felixlândia MG</a></li>
                        <li><a href="${basePath}fotos/fotos-jacaraipe.html" class="mobile-nav-link">Colônia de Férias Jacaraípe ES</a></li>
                        <li><a href="${basePath}fotos/fotos-nova-ponte.html" class="mobile-nav-link">Colônia de Férias Nova Ponte MG</a></li>
                        <li><a href="${basePath}fotos/fotos-clube-aeu.html" class="mobile-nav-link">Clube em Parceria com AEU</a></li>
                    </ul>
                </li>
                <li class="mobile-nav-item">
                    <a href="${basePath}contato.html" class="mobile-nav-link">Fale Conosco</a>
                </li>
            </ul>
        </div>
    `;

    // Adiciona o menu mobile ao final do body
    document.body.insertAdjacentHTML('beforeend', mobileMenuHTML);

    // Verificação final - conta os itens realmente inseridos no DOM
    setTimeout(() => {
        const submenuLinks = document.querySelectorAll('.mobile-nav-submenu .mobile-nav-link');
        const acordosSubmenu = document.querySelector('.mobile-nav-dropdown:nth-child(4) .mobile-nav-submenu');
        if (acordosSubmenu) {
            const acordosLinks = acordosSubmenu.querySelectorAll('.mobile-nav-link');
            console.log(`🔍 Verificação DOM: ${acordosLinks.length} itens realmente inseridos no submenu Acordos`);
            acordosLinks.forEach((link, index) => {
                console.log(`📌 DOM Item ${index + 1}: ${link.textContent} -> ${link.href}`);
            });
        }
    }, 100);

    console.log('✅ Estrutura do menu mobile criada com sucesso');
    console.log('📋 Submenu Acordos e Convenções contém:', acordosSubmenuItems.length, 'itens');
}

/**
 * Inicializa o carrossel de imagens
 */
function initCarousel() {
    const slides = document.querySelectorAll('.carousel-slide');
    const prevBtn = document.getElementById('carousel-prev');
    const nextBtn = document.getElementById('carousel-next');

    if (slides.length === 0) return;

    let currentSlide = 0;
    const totalSlides = slides.length;

    // Função para mostrar um slide específico
    function showSlide(index) {
        // Remove a classe active de todos os slides
        slides.forEach(slide => slide.classList.remove('active'));

        // Adiciona a classe active ao slide atual
        slides[index].classList.add('active');

        currentSlide = index;
    }

    // Função para ir para o próximo slide
    function nextSlide() {
        const next = (currentSlide + 1) % totalSlides;
        showSlide(next);
    }

    // Função para ir para o slide anterior
    function prevSlide() {
        const prev = (currentSlide - 1 + totalSlides) % totalSlides;
        showSlide(prev);
    }

    // Configura as setas de navegação
    if (prevBtn) {
        prevBtn.addEventListener('click', function() {
            prevSlide();
            // Para o autoplay temporariamente quando o usuário interage
            clearInterval(autoplayInterval);
            startAutoplay();
        });
    }

    if (nextBtn) {
        nextBtn.addEventListener('click', function() {
            nextSlide();
            // Para o autoplay temporariamente quando o usuário interage
            clearInterval(autoplayInterval);
            startAutoplay();
        });
    }

    // Autoplay do carrossel
    let autoplayInterval;

    function startAutoplay() {
        autoplayInterval = setInterval(nextSlide, 5000); // Muda a cada 5 segundos
    }

    // Inicia o autoplay
    startAutoplay();

    // Pausa o autoplay quando o mouse está sobre o carrossel
    const carousel = document.querySelector('.carousel');
    if (carousel) {
        carousel.addEventListener('mouseenter', function() {
            clearInterval(autoplayInterval);
        });

        carousel.addEventListener('mouseleave', function() {
            startAutoplay();
        });
    }

    // Pausa o autoplay quando a aba não está visível
    document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
            clearInterval(autoplayInterval);
        } else {
            startAutoplay();
        }
    });
}

/**
 * Inicializa o carrossel de benefícios
 */
function initBeneficiosCarousel() {
    const carousel = document.getElementById('beneficios-carousel');
    const indicators = document.querySelectorAll('.beneficios-indicators .indicator');

    if (!carousel) return;

    let currentSlide = 0;
    const totalSlides = document.querySelectorAll('.beneficio-slide').length;

    if (totalSlides === 0) return;

    // Função para mostrar um slide específico
    function showBeneficioSlide(index) {
        carousel.style.transform = `translateX(-${index * 100}%)`;

        // Atualiza indicadores
        indicators.forEach((indicator, i) => {
            indicator.classList.toggle('active', i === index);
        });

        currentSlide = index;
    }

    // Função para ir para o próximo slide
    function nextBeneficioSlide() {
        const next = (currentSlide + 1) % totalSlides;
        showBeneficioSlide(next);
    }

    // Configura os indicadores
    indicators.forEach((indicator, index) => {
        indicator.addEventListener('click', function() {
            showBeneficioSlide(index);
            clearInterval(beneficiosAutoplayInterval);
            startBeneficiosAutoplay();
        });
    });

    // Autoplay do carrossel
    let beneficiosAutoplayInterval;

    function startBeneficiosAutoplay() {
        beneficiosAutoplayInterval = setInterval(nextBeneficioSlide, 5000); // Muda a cada 5 segundos
    }

    // Inicia o autoplay
    startBeneficiosAutoplay();

    // Pausa o autoplay quando o mouse está sobre o carrossel
    const carouselContainer = document.querySelector('.beneficios-carousel-container');
    if (carouselContainer) {
        carouselContainer.addEventListener('mouseenter', function() {
            clearInterval(beneficiosAutoplayInterval);
        });

        carouselContainer.addEventListener('mouseleave', function() {
            startBeneficiosAutoplay();
        });
    }

    // Suporte a touch/swipe em dispositivos móveis
    let startX = 0;
    let isDragging = false;

    if (carouselContainer) {
        carouselContainer.addEventListener('touchstart', function(e) {
            startX = e.touches[0].clientX;
            isDragging = true;
            clearInterval(beneficiosAutoplayInterval);
        });

        carouselContainer.addEventListener('touchmove', function(e) {
            if (!isDragging) return;
            e.preventDefault();
        });

        carouselContainer.addEventListener('touchend', function(e) {
            if (!isDragging) return;
            isDragging = false;

            const endX = e.changedTouches[0].clientX;
            const diff = startX - endX;

            // Se o swipe foi maior que 50px
            if (Math.abs(diff) > 50) {
                if (diff > 0) {
                    // Swipe para a esquerda - próximo slide
                    nextBeneficioSlide();
                } else {
                    // Swipe para a direita - slide anterior
                    const prev = (currentSlide - 1 + totalSlides) % totalSlides;
                    showBeneficioSlide(prev);
                }
            }

            startBeneficiosAutoplay();
        });
    }

    // Pausa o autoplay quando a aba não está visível
    document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
            clearInterval(beneficiosAutoplayInterval);
        } else {
            startBeneficiosAutoplay();
        }
    });
}

/**
 * Atualiza o ano atual no rodapé
 */
function updateCurrentYear() {
    const yearElement = document.getElementById('current-year');
    if (yearElement) {
        yearElement.textContent = new Date().getFullYear();
    }
}

/**
 * Inicializa a navegação suave entre seções
 */
function initSmoothScrolling() {
    const navLinks = document.querySelectorAll('.navbar-menu a[href^="#"]');

    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();

            const targetId = this.getAttribute('href').substring(1);
            const targetElement = document.getElementById(targetId);

            if (targetElement) {
                // Calcula a posição considerando a altura da navbar fixa
                const navbarHeight = document.querySelector('.navbar').offsetHeight;
                const targetPosition = targetElement.offsetTop - navbarHeight;

                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
            }

            // Atualiza o link ativo
            updateActiveNavLink(this);
        });
    });

    // Atualiza o link ativo baseado na rolagem
    window.addEventListener('scroll', function() {
        updateActiveNavLinkOnScroll();
    });
}

/**
 * Atualiza o link ativo na navegação
 */
function updateActiveNavLink(activeLink) {
    const navLinks = document.querySelectorAll('.navbar-menu a');
    navLinks.forEach(link => link.classList.remove('active'));
    activeLink.classList.add('active');
}

/**
 * Atualiza o link ativo baseado na posição da rolagem
 */
function updateActiveNavLinkOnScroll() {
    const sections = document.querySelectorAll('section[id], main[id]');
    const navLinks = document.querySelectorAll('.navbar-menu a[href^="#"]');
    const navbarHeight = document.querySelector('.navbar').offsetHeight;

    let currentSection = '';

    sections.forEach(section => {
        const sectionTop = section.offsetTop - navbarHeight - 100;
        const sectionHeight = section.offsetHeight;

        if (window.scrollY >= sectionTop && window.scrollY < sectionTop + sectionHeight) {
            currentSection = section.getAttribute('id');
        }
    });

    // Se estamos no topo da página, considera a home como ativa
    if (window.scrollY < 100) {
        currentSection = 'home';
    }

    navLinks.forEach(link => {
        link.classList.remove('active');
        if (link.getAttribute('href') === `#${currentSection}`) {
            link.classList.add('active');
        }
    });
}

/**
 * Função utilitária para debounce (evita execuções excessivas)
 */
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Aplica debounce no evento de scroll para melhor performance
const debouncedScrollHandler = debounce(updateActiveNavLinkOnScroll, 100);
window.addEventListener('scroll', debouncedScrollHandler);

// Função para detectar se está rodando localmente (duplicada para compatibilidade)
function isLocalEnvironmentHome() {
    return window.location.protocol === 'file:' || 
           window.location.hostname === 'localhost' || 
           window.location.hostname === '127.0.0.1' ||
           window.location.hostname === '';
}

// Função para carregar notícias na página inicial
async function carregarNoticiasHome() {
    try {
        console.log('🏠 Carregando notícias para a página inicial...');

        // Verifica se está rodando localmente
        if (isLocalEnvironmentHome()) {
            console.log('🏠 Ambiente local detectado, usando notícias estáticas na home');
            carregarNoticiasEstaticasHome();
            return;
        }

        // Primeiro tenta carregar do Google Sheets (apenas online)
        try {
            console.log('🌐 Ambiente online, tentando carregar do Google Sheets na home...');
            const GOOGLE_SHEETS_URL = `https://docs.google.com/spreadsheets/d/11GrxZcfsZzNYoLz0O65s_HSe-VTufYyo0CkElPW2zMQ/export?format=csv`;
            const response = await fetch(GOOGLE_SHEETS_URL);
            if (response.ok) {
                const csvData = await response.text();
                const noticiasGoogleSheets = parseGoogleSheetsCSVHome(csvData);

                if (noticiasGoogleSheets.length > 0) {
                    // Ordena por data (mais recentes primeiro)
                    noticiasGoogleSheets.sort((a, b) => {
                        if (!a.data || !b.data) return 0;
                        const [diaA, mesA, anoA] = a.data.split('/');
                        const [diaB, mesB, anoB] = b.data.split('/');
                        const dataA = new Date(anoA, mesA - 1, diaA);
                        const dataB = new Date(anoB, mesB - 1, diaB);
                        return dataB - dataA;
                    });

                    // Pega as 3 mais recentes
                    const noticiasRecentes = noticiasGoogleSheets.slice(0, 3);
                    atualizarNoticiasHome(noticiasRecentes);
                    console.log('✅ Notícias carregadas do Google Sheets na home:', noticiasRecentes.length);
                    return;
                }
            }
        } catch (googleSheetsError) {
            console.warn('⚠️ Erro ao carregar do Google Sheets na home, usando notícias estáticas:', googleSheetsError);
        }

        // Fallback: Usa notícias básicas se Google Sheets falhar
        carregarNoticiasEstaticasHome();

    } catch (error) {
        console.error('Erro ao carregar notícias para a home:', error);
        carregarNoticiasEstaticasHome();
    }
}

// Função para carregar notícias estáticas na home
function carregarNoticiasEstaticasHome() {
    const noticiasEstaticas = [
        {
            id: 'noticia-home-1',
            data: '27/03/2025',
            titulo: 'ATENÇÃO, ATENÇÃO',
            categoria: 'CONVOCAÇÃO',
            descricao: 'Dia 31/03/2025 Segunda-feira - Primeira chamada às 18:00, Segunda chamada às 18:30. O Sindicato dos Securitários de MG convida todos os securitários para participarem da Assembleia de Prestação de Contas referente ao ano de 2024.',
            imagem: 'attached_assets/cropimage-20190117180000_1750362489712.jpeg'
        },
        {
            id: 'noticia-home-2',
            data: '26/03/2025',
            titulo: 'CAMPANHA SALARIAL 2025',
            categoria: 'CAMPANHA',
            descricao: 'O Sindicato dos Securitários conquistou uma importante vitória para a categoria na Campanha Salarial 2025.',
            imagem: 'attached_assets/cropimage-20190117180339_1750362489712.jpeg'
        },
        {
            id: 'noticia-home-3',
            data: '20/02/2025',
            titulo: 'AGORA, A LUTA É POR UMA CCT COM GANHOS REAIS!',
            categoria: 'ACORDO',
            descricao: 'PLR DA BRADESCO SEGUROS FOI ASSINADA. AGORA, A LUTA É POR UMA CCT COM GANHOS REAIS!',
            imagem: 'attached_assets/cropimage-20190118112927_1750362489711.jpeg'
        }
    ];

    // Pega as 3 mais recentes
    const noticiasRecentes = noticiasEstaticas.slice(0, 3);
    atualizarNoticiasHome(noticiasRecentes);

    if (isLocalEnvironmentHome()) {
        console.log('✅ Notícias estáticas carregadas para visualização local na home');
    } else {
        console.log('📋 Notícias estáticas carregadas como fallback na home');
    }
}

// Função para processar CSV especificamente para a home (usa mesma lógica da página de notícias)
function parseGoogleSheetsCSVHome(csvData) {
    const noticias = [];

    // Primeiro, processa o CSV considerando células com quebras de linha
    const processedLines = processCSVWithMultilineHome(csvData);
    const lines = processedLines.filter(line => line.trim() !== '');

    console.log('🏠 Processando CSV para home, linhas encontradas:', lines.length);

    // Identifica se há cabeçalho verificando a primeira linha
    let startIndex = 0;
    if (lines.length > 0) {
        const firstLine = lines[0].toLowerCase();
        if (firstLine.includes('data') || firstLine.includes('titulo') || firstLine.includes('título') || firstLine.includes('noticia')) {
            startIndex = 1;
            console.log('🏠 Cabeçalho detectado, iniciando na linha 2');
        }
    }

    // Regex para validar formato de data
    const dateRegex = /^\d{1,2}\/\d{1,2}\/\d{4}$/;

    for (let i = startIndex; i < lines.length; i++) {
        const line = lines[i].trim();
        if (line && line.length > 10) {
            const columns = parseCSVLineHome(line);
            console.log(`🏠 Linha ${i + 1}: ${columns.length} colunas`);

            // Função para validar se uma string é uma data válida
            function isValidDate(dateString) {
                if (!dateString || typeof dateString !== 'string') return false;
                if (!dateRegex.test(dateString.trim())) return false;
                const [dia, mes, ano] = dateString.trim().split('/');
                const date = new Date(ano, mes - 1, dia);
                return date.getFullYear() == ano && 
                       date.getMonth() == (mes - 1) && 
                       date.getDate() == dia;
            }

            // Função para validar se uma string parece um título válido
            function isValidTitle(titleString) {
                if (!titleString || typeof titleString !== 'string') return false;
                const cleaned = titleString.trim();
                return cleaned.length >= 3 && 
                       !dateRegex.test(cleaned) && 
                       !/^\d+$/.test(cleaned);
            }

            let data, titulo, descricao, foto;
            let found = false;

            // Procura pela data e título válidos em qualquer posição
            for (let j = 0; j < Math.min(columns.length, 5); j++) {
                if (found) break;

                if (isValidDate(columns[j])) {
                    for (let k = j + 1; k < Math.min(columns.length, j + 4); k++) {
                        if (isValidTitle(columns[k])) {
                            data = columns[j].trim();
                            titulo = columns[k].trim();

                            // Busca descrição e foto nas colunas seguintes
                            if (k + 1 < columns.length && columns[k + 1] && columns[k + 1].trim() !== '') {
                                descricao = columns[k + 1].trim();
                            }
                            if (k + 2 < columns.length && columns[k + 2] && columns[k + 2].trim() !== '') {
                                foto = columns[k + 2].trim();
                            }

                            found = true;
                            console.log(`🏠 Estrutura detectada na linha ${i + 1}: data=${j}, titulo=${k}`);
                            break;
                        }
                    }
                }
            }

            // Se não encontrou com a busca inteligente, tenta estruturas fixas como fallback
            if (!found && columns.length >= 2) {
                if (isValidDate(columns[0]) && isValidTitle(columns[1])) {
                    data = columns[0].trim();
                    titulo = columns[1].trim();
                    descricao = columns[2] ? columns[2].trim() : '';
                    foto = columns[3] ? columns[3].trim() : '';
                    found = true;
                }
                else if (columns.length >= 3 && isValidDate(columns[1]) && isValidTitle(columns[2])) {
                    data = columns[1].trim();
                    titulo = columns[2].trim();
                    descricao = columns[3] ? columns[3].trim() : '';
                    foto = columns[4] ? columns[4].trim() : '';
                    found = true;
                }
            }

            if (found && data && titulo && data.trim() !== '' && titulo.trim() !== '') {
                const noticiaId = 'noticia-sheets-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9);                const noticia = {
                    id: noticiaId,
                    data: data,
                    titulo: titulo,
                    descricao: descricao || 'Descrição não disponível',
                    imagem: processarURLImagemHome(foto),
                    categoria: 'NOTÍCIA'
                };

                noticias.push(noticia);
                console.log(`🏠 ✅ Notícia adicionada: "${titulo}" (${data}) com ID: ${noticiaId}`);
            }
        }
    }

    console.log('🏠 Total de notícias processadas para home:', noticias.length);
    return noticias;
}

// Função para processar linha CSV (lida com vírgulas dentro de aspas e quebras de linha)
function parseCSVLineHome(line) {
    const result = [];
    let current = '';
    let inQuotes = false;
    let quoteChar = null;

    for (let i = 0; i < line.length; i++) {
        const char = line[i];

        if ((char === '"' || char === "'") && !inQuotes) {
            inQuotes = true;
            quoteChar = char;
        } else if (char === quoteChar && inQuotes) {
            if (i + 1 < line.length && line[i + 1] === quoteChar) {
                current += char;
                i++;
            } else {
                inQuotes = false;
                quoteChar = null;
            }
        } else if (char === ',' && !inQuotes) {
            result.push(current.trim());
            current = '';
        } else {
            current += char;
        }
    }

    result.push(current.trim());

    return result.map(col => {
        let cleaned = col.trim();
        if ((cleaned.startsWith('"') && cleaned.endsWith('"')) || 
            (cleaned.startsWith("'") && cleaned.endsWith("'"))) {
            cleaned = cleaned.slice(1, -1);
        }
        // Converte quebras de linha em HTML, preservando formatação
        cleaned = cleaned.replace(/\\n/g, '<br>').replace(/\r?\n/g, '<br>');
        return cleaned.trim();
    });
}

// Função para processar URL da imagem do Google Drive (para home)
function processarURLImagemHome(urlString) {
    if (!urlString || urlString.trim() === '') {
        return 'attached_assets/logo-20181210104822_1750362776018.png';
    }

    let url = urlString.trim();
    url = url.replace(/^["']|["']$/g, '');

    // Remove função HIPERLINK se existir
    if (url.includes('=HIPERLINK(')) {
        const match = url.match(/=HIPERLINK\("([^"]+)"/);
        if (match && match[1]) {
            url = match[1];
        }
    }

    let fileId = null;

    // Formato: https://drive.google.com/open?id=ABC123
    if (url.includes('drive.google.com/open?id=')) {
        const match = url.match(/[?&]id=([a-zA-Z0-9-_]+)/);
        if (match && match[1]) {
            fileId = match[1];
        }
    }
    // Formato: https://drive.google.com/file/d/ABC123/view
    else if (url.includes('drive.google.com/file/d/')) {
        const match = url.match(/\/file\/d\/([a-zA-Z0-9-_]+)/);
        if (match && match[1]) {
            fileId = match[1];
        }
    }
    // Formato: https://drive.google.com/uc?id=ABC123
    else if (url.includes('drive.google.com/uc?')) {
        const match = url.match(/[?&]id=([a-zA-Z0-9-_]+)/);
        if (match && match[1]) {
            fileId = match[1];
        }
    }

    // Se encontrou um ID do Google Drive, gera URL otimizada
    if (fileId) {
        return `https://drive.google.com/uc?export=view&id=${fileId}`;
    }

    return url || 'attached_assets/logo-20181210104822_1750362776018.png';
}

// Função para atualizar as notícias na página inicial
function atualizarNoticiasHome(noticias) {
    const noticiasGrid = document.querySelector('.noticias-grid');
    if (!noticiasGrid || noticias.length === 0) return;

    // Armazena as notícias globalmente para usar na expansão
    noticiasCarregadas = [...noticias];

    function formatarDataHome(dataString) {
        if (!dataString || !dataString.includes('/')) return '';
        const parts = dataString.split('/');
        if (parts.length !== 3) return '';

        const meses = ['JAN', 'FEV', 'MAR', 'ABR', 'MAI', 'JUN',
            'JUL', 'AGO', 'SET', 'OUT', 'NOV', 'DEZ'];
        const day = parseInt(parts[0]);
        const month = parseInt(parts[1]) - 1;
        const year = parts[2];

        return `${day} ${meses[month]} ${year}`;
    }

    function processarImagemHome(imagem) {
        if (!imagem || imagem.trim() === '' || imagem === 'attached_assets/logo-20181210104822_1750362776018.png') {
            return {
                temImagem: false,
                url: ''
            };
        }

        return {
            temImagem: true,
            url: imagem
        };
    }

    const noticiasHTML = noticias.map(noticia => {
        const imagemInfo = processarImagemHome(noticia.imagem);

        return `
            <div class="noticia-card" onclick="abrirNoticiaHome('${noticia.id}')" style="cursor: pointer;">
                <div class="noticia-imagem-home">
                    ${imagemInfo.temImagem ? 
                        `<img src="${imagemInfo.url}" alt="${noticia.titulo}" data-original-url="${noticia.imagem || ''}" onerror="handleImageErrorHome(this, '${noticia.titulo}')">` : 
                        `<div class="noticia-placeholder">Sem imagem</div>`
                    }
                </div>
                <div class="noticia-conteudo-home">
                    <div class="noticia-data">${formatarDataHome(noticia.data)}</div>
                    <h3>${noticia.titulo}</h3>
                    <p class="noticia-desc">${noticia.descricao ? (noticia.descricao.length > 100 ? noticia.descricao.substring(0, 100) + '...' : noticia.descricao) : ''}</p>
                </div>
            </div>
        `;
    }).join('');

    noticiasGrid.innerHTML = noticiasHTML;
}

// Função para tratar erro de carregamento de imagens na página inicial
function handleImageErrorHome(img, titulo) {
    console.error('❌ Erro ao carregar imagem na home para:', titulo);
    console.error('🔗 URL que falhou:', img.src);

    // Se já tentou a imagem padrão, mostra placeholder
    if (img.src.includes('logo-20181210104822_1750362776018.png')) {
        img.parentElement.innerHTML = '<div class="noticia-placeholder">Sem imagem</div>';
        return;
    }

    // Obtém o URL original se for do Google Drive
    const originalUrl = img.getAttribute('data-original-url') || img.src;
    let fileId = null;

    // Extrai ID do Google Drive
    if (originalUrl.includes('drive.google.com')) {
        const patterns = [
            /[?&]id=([a-zA-Z0-9-_]+)/,
            /\/file\/d\/([a-zA-Z0-9-_]+)/,
            /\/d\/([a-zA-Z0-9-_]+)/
        ];

        for (const pattern of patterns) {
            const match = originalUrl.match(pattern);
            if (match && match[1]) {
                fileId = match[1];
                break;
            }
        }
    }

    // Se tem ID, tenta formatos alternativos
    if (fileId && !img.hasAttribute('data-tried-alternatives-home')) {
        img.setAttribute('data-tried-alternatives-home', 'true');

        const alternativeFormats = [
            `https://drive.usercontent.google.com/download?id=${fileId}&export=view`,
            `https://lh3.googleusercontent.com/d/${fileId}`,
            `https://drive.google.com/uc?id=${fileId}`,
            `https://drive.google.com/thumbnail?id=${fileId}&sz=w400`,
            'attached_assets/cropimage-20190117180000_1750362489712.jpeg'
        ];

        console.log('🔄 Tentando formatos alternativos na home para:', titulo);
        tryAlternativeFormatsHome(img, alternativeFormats, 0, titulo);
    } else {
        // Tenta uma imagem padrão que existe
        img.onerror = function() {
            this.parentElement.innerHTML = '<div class="noticia-placeholder">Sem imagem</div>';
        };
        img.src = 'attached_assets/cropimage-20190117180000_1750362489712.jpeg';
        console.log('🔄 Usando imagem padrão na home para:', titulo);
    }
}

// Função para tentar formatos alternativos na página inicial
function tryAlternativeFormatsHome(img, formats, index, titulo) {
    if (index >= formats.length) {
        // Se todos os formatos falharam, mostra placeholder
        img.parentElement.innerHTML = '<div class="noticia-placeholder">Sem imagem</div>';
        console.log('🔄 Todos os formatos falharam na home, usando placeholder para:', titulo);
        return;
    }

    const currentFormat = formats[index];
    console.log(`🔄 Tentando formato ${index + 1}/${formats.length} na home:`, currentFormat);

    img.onerror = function() {
        console.error(`❌ Formato ${index + 1} falhou na home:`, currentFormat);
        tryAlternativeFormatsHome(img, formats, index + 1, titulo);
    };

    img.onload = function() {
        console.log('✅ Sucesso com formato na home:', currentFormat);
        img.onerror = null;
        img.onload = null;
    };

    img.src = currentFormat;
}

// Função para processar CSV com células que contêm quebras de linha
function processCSVWithMultilineHome(csvData) {
    const lines = [];
    const rawLines = csvData.split('\n');
    let currentLine = '';
    let inQuotes = false;
    let quoteChar = null;

    for (let i = 0; i < rawLines.length; i++) {
        const line = rawLines[i];

        for (let j = 0; j < line.length; j++) {
            const char = line[j];

            if ((char === '"' || char === "'") && !inQuotes) {
                inQuotes = true;
                quoteChar = char;
                currentLine += char;
            } else if (char === quoteChar && inQuotes) {
                if (j + 1 < line.length && line[j + 1] === quoteChar) {
                    currentLine += char + char;
                    j++; // Pula a próxima aspa
                } else {
                    inQuotes = false;
                    quoteChar = null;
                    currentLine += char;
                }
            } else {
                currentLine += char;
            }
        }

        // Se não estamos dentro de aspas, a linha está completa
        if (!inQuotes) {
            if (currentLine.trim() !== '') {
                lines.push(currentLine);
            }
            currentLine = '';
        } else {
            // Se estamos dentro de aspas, adiciona quebra de linha e continua
            currentLine += '\\n';
        }
    }

    // Adiciona a última linha se houver
    if (currentLine.trim() !== '') {
        lines.push(currentLine);
    }

    return lines;
}

// Função para tratar erro de carregamento de imagens no modal expandido da home
function handleImageErrorExpandidoHome(img, titulo) {
    console.error('❌ Erro ao carregar imagem expandida na home para:', titulo);
    console.error('🔗 URL que falhou:', img.src);

    // Obtém o URL original se for do Google Drive
    const originalUrl = img.getAttribute('data-original-url') || img.src;
    let fileId = null;

    // Extrai ID do Google Drive
    if (originalUrl.includes('drive.google.com')) {
        const patterns = [
            /[?&]id=([a-zA-Z0-9-_]+)/,
            /\/file\/d\/([a-zA-Z0-9-_]+)/,
            /\/d\/([a-zA-Z0-9-_]+)/
        ];

        for (const pattern of patterns) {
            const match = originalUrl.match(pattern);
            if (match && match[1]) {
                fileId = match[1];
                break;
            }
        }
    }

    // Se tem ID, tenta formatos alternativos
    if (fileId && !img.hasAttribute('data-tried-alternatives-expandido-home')) {
        img.setAttribute('data-tried-alternatives-expandido-home', 'true');

        const alternativeFormats = [
            `https://drive.usercontent.google.com/download?id=${fileId}&export=view`,
            `https://lh3.googleusercontent.com/d/${fileId}`,
            `https://drive.google.com/uc?id=${fileId}`,
            `https://drive.google.com/thumbnail?id=${fileId}&sz=w800`
        ];

        console.log('🔄 Tentando formatos alternativos no modal da home para:', titulo);
        tryAlternativeFormatsExpandidoHome(img, alternativeFormats, 0, titulo);
    } else {
        // Remove a imagem se não conseguir carregar
        const imagemContainer = img.parentElement;
        if (imagemContainer && imagemContainer.classList.contains('noticia-imagem-expandido')) {
            imagemContainer.style.display = 'none';
        }
        console.log('🔄 Ocultando imagem no modal da home para:', titulo);
    }
}

// Função para tentar formatos alternativos no modal expandido da home
function tryAlternativeFormatsExpandidoHome(img, formats, index, titulo) {
    if (index >= formats.length) {
        const imagemContainer = img.parentElement;
        if (imagemContainer && imagemContainer.classList.contains('noticia-imagem-expandido')) {
            imagemContainer.style.display = 'none';
        }
        console.log('🔄 Todos os formatos falharam no modal da home, ocultando imagem para:', titulo);
        return;
    }

    const currentFormat = formats[index];
    console.log(`🔄 Tentando formato ${index + 1}/${formats.length} no modal da home:`, currentFormat);

    img.onerror = function() {
        console.error(`❌ Formato ${index + 1} falhou no modal da home:`, currentFormat);
        tryAlternativeFormatsExpandidoHome(img, formats, index + 1, titulo);
    };

    img.onload = function() {
        console.log('✅ Sucesso com formato no modal da home:', currentFormat);
        img.onerror = null;
        img.onload = null;
    };

    img.src = currentFormat;
}

// Variável global para armazenar as notícias carregadas
let noticiasCarregadas = [];

function abrirNoticiaHome(noticiaId) {
    console.log('🏠 Expandindo notícia com ID:', noticiaId);

    // Primeiro tenta encontrar nos dados carregados
    let noticiaEncontrada = noticiasCarregadas.find(n => n.id === noticiaId);

    if (!noticiaEncontrada) {
        // Se não encontrou, extrai dados do card clicado
        const card = event.currentTarget;

        if (card) {
            const titulo = card.querySelector('h3') ? card.querySelector('h3').textContent : '';
            const data = card.querySelector('.noticia-data') ? card.querySelector('.noticia-data').textContent : '';
            const descricaoElement = card.querySelector('.noticia-desc');
            let descricao = '';

            if (descricaoElement) {
                descricao = descricaoElement.textContent;
                // Remove os "..." se existirem e expande a descrição
                if (descricao.endsWith('...')) {
                    // Tenta encontrar a descrição completa nos dados carregados
                    const noticiaCompleta = noticiasCarregadas.find(n => n.titulo === titulo);
                    if (noticiaCompleta) {
                        descricao = noticiaCompleta.descricao;
                    } else {
                        descricao = descricao.replace('...', '') + ' Para mais informações, visite nossa sede ou entre em contato conosco.';
                    }
                }
            }

            const imagem = card.querySelector('img') ? card.querySelector('img').src : null;

            noticiaEncontrada = {
                id: noticiaId,
                titulo: titulo,
                data: data,
                descricao: descricao || 'Descrição não disponível',
                imagem: imagem,
                categoria: 'NOTÍCIA'
            };
        } else {
            console.warn('⚠️ Card não encontrado para expandir:', noticiaId);
            return;
        }
    }

    expandirCardNoticia(noticiaEncontrada);
}

// Função para expandir card como modal
function expandirCardNoticia(noticia) {
    // Remove overlay existente se houver
    const overlayExistente = document.querySelector('.noticia-overlay');
    if (overlayExistente) {
        overlayExistente.remove();
    }

    // Cria o overlay
    const overlay = document.createElement('div');
    overlay.className = 'noticia-overlay';

    // Cria o card expandido
    const cardExpandido = document.createElement('div');
    cardExpandido.className = 'noticia-card-expandido';

    // Conteúdo do card expandido
    cardExpandido.innerHTML = `
        <div class="noticia-header-expandido">
            <button class="btn-fechar-noticia" onclick="fecharCardNoticia()">×</button>
            <div class="noticia-data-expandido">${noticia.data}</div>
            <h2 class="noticia-titulo-expandido">${noticia.titulo}</h2>
        </div>

        ${noticia.imagem && !noticia.imagem.includes('logo-20181210104822') ? `
            <div class="noticia-imagem-expandido">
                <img src="${noticia.imagem}" 
                     alt="${noticia.titulo}" 
                     data-original-url="${noticia.imagem}"
                     onerror="handleImageErrorExpandidoHome(this, '${noticia.titulo}')">
            </div>
        ` : ''}

        <div class="noticia-conteudo-expandido">
            <div class="noticia-categoria-expandido">${noticia.categoria}</div>
            <div class="noticia-descricao-expandido">
                ${noticia.descricao}
            </div>
        </div>
    `;

    overlay.appendChild(cardExpandido);
    document.body.appendChild(overlay);

    // Adiciona evento para fechar ao clicar no overlay
    overlay.addEventListener('click', function(e) {
        if (e.target === overlay) {
            fecharCardNoticia();
        }
    });

    // Adiciona evento para fechar com ESC
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            fecharCardNoticia();
        }
    });

    // Anima a entrada
    setTimeout(() => {
        overlay.classList.add('ativo');
        cardExpandido.classList.add('ativo');
    }, 10);
}

// Função para fechar o card expandido
function fecharCardNoticia() {
    const overlay = document.querySelector('.noticia-overlay');
    if (overlay) {
        overlay.classList.remove('ativo');
        setTimeout(() => {
            overlay.remove();
        }, 300);
    }
}

/**
 * Aplicar máscara de telefone nos campos
 */
function aplicarMascaraTelefone() {
    const telefoneInputs = document.querySelectorAll('input[type="tel"], input[name="telefone"]');

    telefoneInputs.forEach(input => {
        input.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');

            if (value.length <= 11) {
                if (value.length <= 10) {
                    value = value.replace(/(\d{2})(\d{4})(\d{4})/, '($1) $2-$3');
                } else {
                    value = value.replace(/(\d{2})(\d{5})(\d{4})/, '($1) $2-$3');
                }
            }

            e.target.value = value;
        });
    });
}

/**
 * Validar formulários básicos
 */
function validarFormularios() {
    const forms = document.querySelectorAll('form');

    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const requiredFields = form.querySelectorAll('[required]');
            let valid = true;

            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    valid = false;
                    field.style.borderColor = '#ff0000';
                } else {
                    field.style.borderColor = '';
                }
            });

            if (!valid) {
                e.preventDefault();
                alert('Por favor, preencha todos os campos obrigatórios.');
            }
        });
    });
}

/**
 * Função básica de envio de email
 */
function enviarEmail() {
    // Função placeholder para compatibilidade
    console.log('Sistema de envio de email carregado');
}

// Define a função configurarEventos
function configurarEventos() {
    // Configurações adicionais de eventos
    console.log('✅ Função configurarEventos() inicializada.');

    // Configurar dropdowns do menu desktop se existirem
    const dropdowns = document.querySelectorAll('.dropdown');
    dropdowns.forEach(dropdown => {
        const link = dropdown.querySelector('a');
        const content = dropdown.querySelector('.dropdown-content');

        if (link && content) {
            dropdown.addEventListener('mouseenter', () => {
                content.style.display = 'block';
            });

            dropdown.addEventListener('mouseleave', () => {
                content.style.display = 'none';
            });
        }
    });
}


  

    // Verifica se as logos foram encontradas
    if (!desktopLogo && !mobileLogo) {
        console.warn('⚠️ Nenhuma logo encontrada para correção');
    }
}

// Aguarda o DOM estar completamente carregado
document.addEventListener('DOMContentLoaded', function() {


    console.log('🔧 Criando estrutura do menu mobile...');
    createMobileMenuStructure();
    initMobileMenu(); // Sistema de menu mobile
    initCarousel(); // Carrossel de imagens
    initBeneficiosCarousel(); // Carrossel de benefícios
    configurarEventos(); // Eventos gerais

    // Carregar notícias apenas na página inicial
    if (isHomePage()) {
        loadNoticiasHome();
    }

    // Verificação adicional para garantir que o submenu foi criado
    setTimeout(() => {
        const acordosSubmenuItems = document.querySelectorAll('#mobile-acordos-submenu .mobile-submenu-item');
        console.log('🔍 Verificação DOM:', acordosSubmenuItems.length, 'itens realmente inseridos no submenu Acordos');

        acordosSubmenuItems.forEach((item, index) => {
            const link = item.querySelector('a');
            if (link) {
                console.log(`📌 DOM Item ${index + 1}: ${item.textContent.trim()} -> ${link.href}`);
            }
        });
    }, 100);
    
    // Correção adicional da logo após a criação do menu mobile
    setTimeout(() => {
        fixNavbarLogo();
    }, 200);
});